//=============================================================================
// HasItemsSaver_MV.js
//=============================================================================
// RPGツクールMV用に修正
// (C)2021 maguros
// MIT License
//=============================================================================

/*:
 * @plugindesc アイテムの保存・読み出しを行うMV用プラグイン
 * @author maguros
 * 
 * @param ProtectedItemList
 * @text 保護アイテム一覧（アイテム）
 * @desc 上書き・消去の対象にしないアイテムの一覧。
 * @type item[]
 * 
 * @param ProtectedWeaponList
 * @text 保護アイテム一覧（武器）
 * @desc 上書き・消去の対象にしない武器の一覧。
 * @type weapon[]
 * 
 * @param ProtectedArmorList
 * @text 保護アイテム一覧（防具）
 * @desc 上書き・消去の対象にしない防具の一覧。
 * @type armor[]
 * 
 * @help
 * アイテムの保存・読み出しを行うプラグインです。
 * 
 * 【プラグインコマンド】
 * SaveHasItems [変数ID]  : アイテムを変数に保存
 * LoadHasItems [変数ID]  : 変数からアイテムを復元
 * ClearAllHasItems       : アイテムを全消去
 * 
 * 利用規約：
 * 　作者に無断で改変、再配布が可能で、利用形態（商用、18禁利用等）
 * 　についても制限はありません。
 */

(function() {
    'use strict';

    var parameters = PluginManager.parameters('HasItemsSaver_MV');
    var protectedItems = JSON.parse(parameters["ProtectedItemList"] || "[]").map(Number);
    var protectedWeapons = JSON.parse(parameters["ProtectedWeaponList"] || "[]").map(Number);
    var protectedArmors = JSON.parse(parameters["ProtectedArmorList"] || "[]").map(Number);

    function isProtectedItem(itemId, type) {
        switch (type) {
            case 'item': return protectedItems.includes(itemId);
            case 'weapon': return protectedWeapons.includes(itemId);
            case 'armor': return protectedArmors.includes(itemId);
            default: return false;
        }
    }

    function clearItems(type, items) {
        for (var key in items) {
            var itemId = Number(key);
            if (isProtectedItem(itemId, type)) continue;
            $gameParty.loseItem(getItemById(itemId, type), items[key], false);
        }
    }

    function getItemById(itemId, type) {
        switch (type) {
            case 'item': return $dataItems[itemId];
            case 'weapon': return $dataWeapons[itemId];
            case 'armor': return $dataArmors[itemId];
            default: return null;
        }
    }

    function saveHasItems(variableId) {
        if (variableId <= 0) return;
        var itemData = {
            items: Object.assign({}, $gameParty._items),
            weapons: Object.assign({}, $gameParty._weapons),
            armors: Object.assign({}, $gameParty._armors)
        };
        $gameVariables.setValue(variableId, JSON.stringify(itemData));
    }

    function loadHasItems(variableId) {
        if (variableId <= 0) return;

        var json = $gameVariables.value(variableId);
        if (!json) return;

        var itemData = JSON.parse(json);
        $gameParty._items = {};
        $gameParty._weapons = {};
        $gameParty._armors = {};

        for (var key in itemData.items) {
            var itemId = Number(key);
            if (!isProtectedItem(itemId, 'item')) {
                $gameParty.gainItem($dataItems[itemId], itemData.items[key]);
            }
        }

        for (var key in itemData.weapons) {
            var itemId = Number(key);
            if (!isProtectedItem(itemId, 'weapon')) {
                $gameParty.gainItem($dataWeapons[itemId], itemData.weapons[key]);
            }
        }

        for (var key in itemData.armors) {
            var itemId = Number(key);
            if (!isProtectedItem(itemId, 'armor')) {
                $gameParty.gainItem($dataArmors[itemId], itemData.armors[key]);
            }
        }
    }

    function clearAllHasItems() {
        clearItems('item', $gameParty._items);
        clearItems('weapon', $gameParty._weapons);
        clearItems('armor', $gameParty._armors);
    }

    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (command === 'SaveHasItems') {
            var variableId = Number(args[0] || 0);
            saveHasItems(variableId);
        } else if (command === 'LoadHasItems') {
            var variableId = Number(args[0] || 0);
            loadHasItems(variableId);
        } else if (command === 'ClearAllHasItems') {
            clearAllHasItems();
        }
    };
})();